<?php
header('Content-Type: application/json');
require_once 'db.php';

// ── Allowed types ──────────────────────────────────────────────────────────
$IMAGE_MIMES = ['image/jpeg','image/png','image/gif','image/webp'];
$DOC_MIMES   = [
    'application/pdf',
    'application/msword',
    'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
    'application/vnd.ms-excel',
    'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
    'text/plain',
];
$IMAGE_EXTS = ['jpg','jpeg','png','gif','webp'];
$DOC_EXTS   = ['pdf','doc','docx','xls','xlsx','txt'];
$MAX_BYTES  = 10 * 1024 * 1024; // 10 MB

function jsonError(string $msg, int $code = 400): void {
    http_response_code($code);
    echo json_encode(['success' => false, 'error' => $msg]);
    exit;
}

// ── Validate sender ────────────────────────────────────────────────────────
$allowed = ['gihan', 'teah'];
$sender  = strtolower(trim($_POST['sender'] ?? ''));
$caption = trim($_POST['message'] ?? '');

if (!in_array($sender, $allowed)) jsonError('Invalid sender');

// ── Validate file ──────────────────────────────────────────────────────────
if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
    jsonError('No file uploaded or upload error: ' . ($_FILES['file']['error'] ?? 'missing'));
}

$file    = $_FILES['file'];
$origName = basename($file['name']);
$ext     = strtolower(pathinfo($origName, PATHINFO_EXTENSION));
$size    = $file['size'];
$tmpPath = $file['tmp_name'];

if ($size > $MAX_BYTES) jsonError('File too large (max 10 MB)');

// Detect MIME from content (safer than trusting the browser-reported MIME)
$finfo    = new finfo(FILEINFO_MIME_TYPE);
$mime     = $finfo->file($tmpPath);

if (in_array($mime, $IMAGE_MIMES) && in_array($ext, $IMAGE_EXTS)) {
    $fileType = 'image';
} elseif (in_array($mime, $DOC_MIMES) && in_array($ext, $DOC_EXTS)) {
    $fileType = 'document';
} else {
    jsonError('Unsupported file type. Allowed: jpg, png, gif, webp, pdf, doc, docx, xls, xlsx, txt');
}

// ── Save file ──────────────────────────────────────────────────────────────
$uploadDir = __DIR__ . '/uploads/';
if (!is_dir($uploadDir)) {
    mkdir($uploadDir, 0755, true);
}

// Unique filename: timestamp + random + sanitised original name
$safeName  = preg_replace('/[^a-zA-Z0-9._-]/', '_', $origName);
$uniqueName = time() . '_' . bin2hex(random_bytes(4)) . '_' . $safeName;
$destPath   = $uploadDir . $uniqueName;
$webPath    = 'uploads/' . $uniqueName;

if (!move_uploaded_file($tmpPath, $destPath)) {
    jsonError('Failed to save file', 500);
}

// ── Insert message row ────────────────────────────────────────────────────
$stmt = $pdo->prepare(
    'INSERT INTO messages (sender, message, file_path, file_type) VALUES (?, ?, ?, ?)'
);
$stmt->execute([$sender, $caption, $webPath, $fileType]);

echo json_encode([
    'success'   => true,
    'id'        => $pdo->lastInsertId(),
    'file_path' => $webPath,
    'file_type' => $fileType,
]);
