<?php
header('Content-Type: application/json');

$url = trim($_GET['url'] ?? '');

// Basic URL validation
if (!filter_var($url, FILTER_VALIDATE_URL)) {
    echo json_encode(['error' => 'Invalid URL']);
    exit;
}

// Only allow http/https
$scheme = strtolower(parse_url($url, PHP_URL_SCHEME));
if (!in_array($scheme, ['http', 'https'])) {
    echo json_encode(['error' => 'Unsupported scheme']);
    exit;
}

// Fetch the page with a browser-like User-Agent and timeout
$ctx = stream_context_create([
    'http' => [
        'timeout'          => 5,
        'follow_location'  => true,
        'max_redirects'    => 5,
        'header'           => [
            'User-Agent: Mozilla/5.0 (compatible; GTChatBot/1.0)',
            'Accept: text/html,application/xhtml+xml',
        ],
    ],
    'ssl' => [
        'verify_peer'      => false,
        'verify_peer_name' => false,
    ],
]);

$html = @file_get_contents($url, false, $ctx);
if ($html === false) {
    echo json_encode(['error' => 'Could not fetch URL']);
    exit;
}

// Only parse the <head> section to keep it fast
$head = '';
if (preg_match('/<head[^>]*>(.*?)<\/head>/is', $html, $m)) {
    $head = $m[1];
} else {
    $head = substr($html, 0, 8000);
}

// Helper: get meta tag content
function getMeta(string $html, string $property): string {
    // og: / twitter: style  <meta property="og:title" content="...">
    if (preg_match(
        '/<meta[^>]+(?:property|name)\s*=\s*["\']' . preg_quote($property, '/') . '["\'][^>]+content\s*=\s*["\']([^"\']*)["\'][^>]*>/i',
        $html, $m
    )) return html_entity_decode($m[1], ENT_QUOTES, 'UTF-8');

    // reversed attribute order
    if (preg_match(
        '/<meta[^>]+content\s*=\s*["\']([^"\']*)["\'][^>]+(?:property|name)\s*=\s*["\']' . preg_quote($property, '/') . '["\'][^>]*>/i',
        $html, $m
    )) return html_entity_decode($m[1], ENT_QUOTES, 'UTF-8');

    return '';
}

$title       = getMeta($head, 'og:title');
$description = getMeta($head, 'og:description');
$image       = getMeta($head, 'og:image');

// Fallbacks
if (!$title) {
    if (preg_match('/<title[^>]*>([^<]+)<\/title>/i', $head, $m))
        $title = html_entity_decode(trim($m[1]), ENT_QUOTES, 'UTF-8');
}
if (!$description) $description = getMeta($head, 'description');
if (!$image)       $image       = getMeta($head, 'twitter:image');

// Resolve relative image URL
if ($image && !preg_match('/^https?:\/\//i', $image)) {
    $parts = parse_url($url);
    $base  = $parts['scheme'] . '://' . $parts['host'];
    $image = $base . '/' . ltrim($image, '/');
}

// Truncate long strings
$title       = mb_substr($title, 0, 120);
$description = mb_substr($description, 0, 200);

echo json_encode([
    'title'       => $title,
    'description' => $description,
    'image'       => $image,
    'url'         => $url,
]);
