<?php
$allowed_users = ['gihan', 'teah'];
$user = strtolower(trim($_GET['user'] ?? ''));

if (!in_array($user, $allowed_users)) {
    header('Location: index.php');
    exit;
}

$other = ($user === 'gihan') ? 'teah' : 'gihan';
$names = ['gihan' => 'Gihan', 'teah' => 'Teah'];
$userName  = $names[$user];
$otherName = $names[$other];
$userInitial  = strtoupper($user[0]);
$otherInitial = strtoupper($other[0]);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($userName) ?> › GTChat</title>
    <link rel="stylesheet" href="style.css">
    <!-- PeerJS for WebRTC -->
    <script src="https://unpkg.com/peerjs@1.5.2/dist/peerjs.min.js"></script>
    <?php if ($user === 'teah'): ?>
    <style>.chat-input-bar textarea:focus { border-color: var(--teah-color); }</style>
    <?php endif; ?>
</head>
<body>
<div class="chat-wrapper">

    <!-- HEADER -->
    <header class="chat-header header-<?= $user ?>">
        <a href="index.php" class="back-btn" title="Back">&#8592;</a>
        <div class="header-avatar"><?= $userInitial ?></div>
        <div class="header-info">
            <h2><?= htmlspecialchars($userName) ?></h2>
            <p>Chatting with <?= htmlspecialchars($otherName) ?></p>
        </div>
        <div class="header-calls">
            <button class="call-btn" id="voiceCallBtn" title="Voice Call">📞</button>
            <button class="call-btn" id="videoCallBtn" title="Video Call">🎥</button>
        </div>
    </header>

    <!-- MESSAGES -->
    <section class="chat-messages" id="chatMessages">
        <div class="empty-state" id="emptyState">
            <div class="emoji">👋</div>
            <p>No messages yet.<br>Say hello to <?= htmlspecialchars($otherName) ?>!</p>
        </div>
    </section>

    <!-- FILE PREVIEW BAR -->
    <div class="file-preview-bar" id="filePreviewBar" style="display:none;">
        <div class="fp-inner" id="fpInner"></div>
        <button class="fp-remove" id="fpRemove" title="Remove">&#10005;</button>
    </div>

    <!-- INPUT BAR -->
    <div class="chat-input-bar">
        <input type="file" id="fileInput"
            accept="image/jpeg,image/png,image/gif,image/webp,
                    application/pdf,application/msword,
                    application/vnd.openxmlformats-officedocument.wordprocessingml.document,
                    application/vnd.ms-excel,
                    application/vnd.openxmlformats-officedocument.spreadsheetml.sheet,
                    text/plain,
                    .jpg,.jpeg,.png,.gif,.webp,.pdf,.doc,.docx,.xls,.xlsx,.txt"
            style="display:none;">
        <button class="attach-btn attach-btn-<?= $user ?>" id="attachBtn" type="button" title="Attach file">📎</button>
        <textarea id="msgInput" placeholder="Message <?= htmlspecialchars($otherName) ?>..."
            rows="1" maxlength="1000" autofocus></textarea>
        <button class="send-btn send-btn-<?= $user ?>" id="sendBtn" title="Send">&#9658;</button>
    </div>
</div>

<!-- ── LIGHTBOX ──────────────────────────────────────────────────────── -->
<div class="lightbox" id="lightbox" role="dialog" aria-modal="true" aria-label="Image preview">
    <button class="lb-close" id="lbClose" title="Close">&#10005;</button>
    <img class="lb-img" id="lbImg" src="" alt="Preview">
    <div class="lb-caption" id="lbCaption"></div>
</div>

<!-- ── CALL OVERLAY ─────────────────────────────────────────────────── -->
<div class="call-overlay" id="callOverlay">
    <div class="call-content">
        <div class="caller-avatar" id="callAvatar"><?= $otherInitial ?></div>
        <h3 id="callStatus">Calling...</h3>
        <p id="callName"><?= htmlspecialchars($otherName) ?></p>
        
        <div class="video-container" id="videoContainer" style="display:none;">
            <video id="remoteVideo" autoplay playsinline class="remote-video"></video>
            <video id="localVideo" autoplay playsinline muted class="local-video"></video>
        </div>

        <div class="call-actions">
            <button class="action-btn decline" id="declineBtn" title="Decline">✖</button>
            <button class="action-btn answer" id="answerBtn" title="Answer" style="display:none;">✔</button>
        </div>
    </div>
    <!-- Simple ringtone -->
    <audio id="ringtone" loop src="https://assets.mixkit.co/active_storage/sfx/2358/2358-preview.mp3"></audio>
</div>

<script>
const CURRENT_USER = <?= json_encode($user) ?>;
const OTHER_USER   = <?= json_encode($other) ?>;
const PEER_ID      = 'gtchat_v1_' + CURRENT_USER;
const OTHER_PEER_ID= 'gtchat_v1_' + OTHER_USER;

let lastId = 0;
let polling = null;
let selectedFile = null;

// Call state
let myPeer     = null;
let currentCall= null;
let localStream= null;
let isVideoCall= false;

// DOM refs
const chatMessages   = document.getElementById('chatMessages');
const msgInput       = document.getElementById('msgInput');
const sendBtn        = document.getElementById('sendBtn');
const emptyState     = document.getElementById('emptyState');
const attachBtn      = document.getElementById('attachBtn');
const fileInput      = document.getElementById('fileInput');
const filePreviewBar = document.getElementById('filePreviewBar');
const fpInner        = document.getElementById('fpInner');
const fpRemove       = document.getElementById('fpRemove');
const lightbox       = document.getElementById('lightbox');
const lbImg          = document.getElementById('lbImg');
const lbCaption      = document.getElementById('lbCaption');
const lbClose        = document.getElementById('lbClose');

// ── IntersectionObserver — mark received messages as seen ─────────────────
const seenObserver = new IntersectionObserver((entries) => {
    const ids = [];
    entries.forEach(e => {
        if (e.isIntersecting && e.target.dataset.received === '1') {
            ids.push(e.target.dataset.id);
            seenObserver.unobserve(e.target);
        }
    });
    if (ids.length) markSeen(ids);
}, { threshold: 0.6 });

async function markSeen(ids) {
    try {
        const fd = new FormData();
        fd.append('viewer', CURRENT_USER);
        fd.append('ids', ids.join(','));
        await fetch('mark_seen.php', { method: 'POST', body: fd });
    } catch(e) { /* silent */ }
}

// ── Lightbox ───────────────────────────────────────────────────────────────
function openLightbox(src, caption) {
    lbImg.src = src;
    lbCaption.textContent = caption || '';
    lightbox.classList.add('active');
    document.body.style.overflow = 'hidden';
}

function closeLightbox() {
    lightbox.classList.remove('active');
    document.body.style.overflow = '';
    lbImg.src = '';
}

lbClose.addEventListener('click', closeLightbox);
lightbox.addEventListener('click', (e) => { if (e.target === lightbox) closeLightbox(); });
document.addEventListener('keydown', (e) => { if (e.key === 'Escape') closeLightbox(); });

// ── Textarea auto-resize ───────────────────────────────────────────────────
msgInput.addEventListener('input', () => {
    msgInput.style.height = 'auto';
    msgInput.style.height = Math.min(msgInput.scrollHeight, 120) + 'px';
});

msgInput.addEventListener('keydown', (e) => {
    if (e.key === 'Enter' && !e.shiftKey) { e.preventDefault(); sendMessage(); }
});
sendBtn.addEventListener('click', sendMessage);

// ── File attach ────────────────────────────────────────────────────────────
attachBtn.addEventListener('click', () => fileInput.click());
fileInput.addEventListener('change', () => {
    const file = fileInput.files[0];
    if (!file) return;
    selectedFile = file;
    showFilePreview(file);
});
fpRemove.addEventListener('click', clearFile);

function showFilePreview(file) {
    fpInner.innerHTML = '';
    if (file.type.startsWith('image/')) {
        const img = document.createElement('img');
        img.className = 'fp-thumb';
        img.src = URL.createObjectURL(file);
        img.alt = file.name;
        fpInner.appendChild(img);
    }
    const pill = document.createElement('span');
    pill.className = 'fp-pill';
    pill.textContent = file.name + ' (' + formatBytes(file.size) + ')';
    fpInner.appendChild(pill);
    filePreviewBar.style.display = 'flex';
}

function clearFile() {
    selectedFile = null;
    fileInput.value = '';
    filePreviewBar.style.display = 'none';
    fpInner.innerHTML = '';
}

// ── Utilities ──────────────────────────────────────────────────────────────
function formatTime(dateStr) {
    return new Date(dateStr).toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
}

function formatBytes(bytes) {
    if (bytes < 1024) return bytes + ' B';
    if (bytes < 1048576) return (bytes / 1024).toFixed(1) + ' KB';
    return (bytes / 1048576).toFixed(1) + ' MB';
}

function escapeHtml(str) {
    const d = document.createElement('div');
    d.textContent = str;
    return d.innerHTML;
}

// ── URL hyperlinking ───────────────────────────────────────────────────────
const URL_REGEX = /(https?:\/\/[^\s<>"{}|\\^`[\]]+)/gi;
function formatMessageText(str) {
    return escapeHtml(str).replace(URL_REGEX, (url) => {
        const safeUrl = url.replace(/&amp;/g, '&');
        return `<a href="${safeUrl}" target="_blank" rel="noopener noreferrer" class="msg-link">${url}</a>`;
    });
}
function extractFirstUrl(str) {
    const m = str.match(URL_REGEX); return m ? m[0] : null;
}

async function fetchLinkPreview(url, container) {
    try {
        const data = await (await fetch(`link_preview.php?url=${encodeURIComponent(url)}`)).json();
        if (!data || data.error) return;
        const prev = document.createElement('div');
        prev.className = 'link-preview';
        prev.innerHTML = `
            ${data.image ? `<img src="${escapeHtml(data.image)}" class="lp-img" alt="" onerror="this.style.display='none'">` : ''}
            <div class="lp-body">
                ${data.title ? `<div class="lp-title">${escapeHtml(data.title)}</div>` : ''}
                ${data.description ? `<div class="lp-desc">${escapeHtml(data.description)}</div>` : ''}
                <div class="lp-url">${escapeHtml(new URL(url).hostname)}</div>
            </div>`;
        prev.addEventListener('click', () => window.open(url, '_blank', 'noopener'));
        container.appendChild(prev);
    } catch(e) {}
}

// ── Status ticks ───────────────────────────────────────────────────────────
// ✓ = sent, ✓✓ = delivered (grey), ✓✓ = seen (blue)
function buildTick(status) {
    const span = document.createElement('span');
    span.className = `msg-status status-${status}`;
    span.dataset.statusEl = '1';
    if (status === 'sent')      span.innerHTML = '✓';
    if (status === 'delivered') span.innerHTML = '✓✓';
    if (status === 'seen')      span.innerHTML = '✓✓';
    span.title = status.charAt(0).toUpperCase() + status.slice(1);
    return span;
}

function updateRowStatus(row, status) {
    const existing = row.querySelector('[data-status-el]');
    if (existing) existing.remove();
    row.appendChild(buildTick(status));
}

// ── Image bubble ───────────────────────────────────────────────────────────
function buildImageBubble(bubbleClass, filePath, caption) {
    const wrap = document.createElement('div');
    wrap.className = bubbleClass + ' bubble-media';

    const img = document.createElement('img');
    img.src = filePath;
    img.className = 'msg-image';
    img.alt = 'image';
    img.loading = 'lazy';
    img.addEventListener('click', () => openLightbox(filePath, caption || ''));
    wrap.appendChild(img);

    if (caption) {
        const cap = document.createElement('p');
        cap.className = 'media-caption';
        cap.innerHTML = formatMessageText(caption);
        wrap.appendChild(cap);
    }
    return wrap;
}

// ── Document pill ──────────────────────────────────────────────────────────
const DOC_ICONS = { pdf:'📄', doc:'📝', docx:'📝', xls:'📊', xlsx:'📊', txt:'📃' };
function buildDocBubble(bubbleClass, filePath, caption) {
    const wrap = document.createElement('div');
    wrap.className = bubbleClass + ' bubble-media';

    const fileName = filePath.split('/').pop().replace(/^\d+_[0-9a-f]+_/, '');
    const ext = fileName.split('.').pop().toLowerCase();
    const icon = DOC_ICONS[ext] || '📎';

    const pill = document.createElement('a');
    pill.href = filePath;
    pill.target = '_blank';
    pill.rel = 'noopener noreferrer';
    pill.className = 'doc-pill';
    pill.innerHTML = `<span class="doc-icon">${icon}</span>
        <span class="doc-name">${escapeHtml(fileName)}</span>
        <span class="doc-dl">⬇</span>`;
    wrap.appendChild(pill);

    if (caption) {
        const cap = document.createElement('p');
        cap.className = 'media-caption';
        cap.innerHTML = formatMessageText(caption);
        wrap.appendChild(cap);
    }
    return wrap;
}

// ── Append message row ─────────────────────────────────────────────────────
function appendMessage(msg) {
    if (emptyState) emptyState.style.display = 'none';

    const isSent = msg.sender === CURRENT_USER;
    const row = document.createElement('div');
    row.className = 'message-row ' + (isSent ? 'sent' : 'received');
    row.dataset.id = msg.id;
    if (!isSent) row.dataset.received = '1';

    const bubbleClass = isSent ? 'bubble sent-' + CURRENT_USER : 'bubble received';

    if (msg.file_path) {
        const bubble = msg.file_type === 'image'
            ? buildImageBubble(bubbleClass, msg.file_path, msg.message)
            : buildDocBubble(bubbleClass, msg.file_path, msg.message);
        row.appendChild(bubble);
    } else {
        const bubble = document.createElement('div');
        bubble.className = bubbleClass;
        bubble.innerHTML = formatMessageText(msg.message);
        row.appendChild(bubble);
        const firstUrl = extractFirstUrl(msg.message);
        if (firstUrl) fetchLinkPreview(firstUrl, row);
    }

    // Time
    const timeSpan = document.createElement('span');
    timeSpan.className = 'bubble-time';
    timeSpan.textContent = formatTime(msg.sent_at);
    row.appendChild(timeSpan);

    // Status ticks (only on own messages)
    if (isSent && msg.status) row.appendChild(buildTick(msg.status));

    chatMessages.appendChild(row);

    // Observe incoming messages for seen tracking
    if (!isSent) seenObserver.observe(row);
}

function scrollToBottom() { chatMessages.scrollTop = chatMessages.scrollHeight; }

// ── Update statuses of already-rendered sent messages ─────────────────────
function refreshStatuses(messages) {
    messages.forEach(m => {
        if (m.sender !== CURRENT_USER) return;
        const row = chatMessages.querySelector(`[data-id="${m.id}"]`);
        if (row && m.status) updateRowStatus(row, m.status);
    });
}

// ── Call Logic (PeerJS) ────────────────────────────────────────────────────
const callOverlay = document.getElementById('callOverlay');
const callStatus  = document.getElementById('callStatus');
const declineBtn  = document.getElementById('declineBtn');
const answerBtn   = document.getElementById('answerBtn');
const localVideo  = document.getElementById('localVideo');
const remoteVideo = document.getElementById('remoteVideo');
const videoContainer = document.getElementById('videoContainer');
const ringtone    = document.getElementById('ringtone');

function initPeer() {
    myPeer = new Peer(PEER_ID);
    myPeer.on('open', (id) => console.log('My Peer ID:', id));
    myPeer.on('call', (call) => {
        currentCall = call;
        // Signaling already triggered UI, but this handles the stream handshake
    });
    myPeer.on('error', (err) => {
        console.error('Peer error:', err);
        if (err.type === 'peer-unavailable') {
            alert(OTHER_USER + ' is offline or not in chat.');
            endCall();
        }
    });
}

async function startCall(video) {
    isVideoCall = video;
    try {
        localStream = await navigator.mediaDevices.getUserMedia({ audio: true, video: video });
        localVideo.srcObject = localStream;
        videoContainer.style.display = video ? 'block' : 'none';
        showCallUI('outgoing');
        
        // Signal the other user to ring
        const signal = video ? '__CALL_SIGNAL__:VIDEO__' : '__CALL_SIGNAL__:VOICE__';
        const fd = new FormData();
        fd.append('sender', CURRENT_USER);
        fd.append('message', signal);
        await fetch('send_message.php', { method: 'POST', body: fd });

        const call = myPeer.call(OTHER_PEER_ID, localStream);
        setupCallHandlers(call);
    } catch (err) {
        console.error('Call failed:', err);
        alert('Camera/Mic access denied or error: ' + err.message);
    }
}

function setupCallHandlers(call) {
    currentCall = call;
    call.on('stream', (remoteStream) => {
        remoteVideo.srcObject = remoteStream;
        callStatus.textContent = 'In Call';
        ringtone.pause();
    });
    call.on('close', endCall);
    call.on('error', endCall);
}

function showCallUI(type) {
    callOverlay.classList.add('active');
    if (type === 'outgoing') {
        callStatus.textContent = 'Calling...';
        answerBtn.style.display = 'none';
    } else if (type === 'incoming') {
        callStatus.textContent = isVideoCall ? 'Incoming Video Call...' : 'Incoming Voice Call...';
        answerBtn.style.display = 'flex';
        ringtone.play().catch(e => {});
    }
}

async function answerCall() {
    answerBtn.style.display = 'none';
    ringtone.pause();
    try {
        localStream = await navigator.mediaDevices.getUserMedia({ audio: true, video: isVideoCall });
        localVideo.srcObject = localStream;
        videoContainer.style.display = isVideoCall ? 'block' : 'none';
        
        currentCall.answer(localStream);
        setupCallHandlers(currentCall);
    } catch (err) {
        console.error('Answer failed:', err);
        endCall();
    }
}

async function endCall() {
    if (currentCall) currentCall.close();
    if (localStream) localStream.getTracks().forEach(t => t.stop());
    
    currentCall = null;
    localStream = null;
    callOverlay.classList.remove('active');
    videoContainer.style.display = 'none';
    ringtone.pause();

    // Notify other person if they haven't ended yet
    const fd = new FormData();
    fd.append('sender', CURRENT_USER);
    fd.append('message', '__CALL_SIGNAL__:END__');
    await fetch('send_message.php', { method: 'POST', body: fd });
}

declineBtn.addEventListener('click', endCall);
answerBtn.addEventListener('click', answerCall);
document.getElementById('voiceCallBtn').addEventListener('click', () => startCall(false));
document.getElementById('videoCallBtn').addEventListener('click', () => startCall(true));

// ── Send ───────────────────────────────────────────────────────────────────
async function sendMessage() {
    const text = msgInput.value.trim();
    if (!text && !selectedFile) return;

    const fileToSend = selectedFile;
    msgInput.value = '';
    msgInput.style.height = 'auto';
    sendBtn.disabled = true;
    clearFile();

    try {
        const fd = new FormData();
        fd.append('sender', CURRENT_USER);

        if (fileToSend) {
            fd.append('file', fileToSend);
            fd.append('message', text);
            const data = await (await fetch('upload.php', { method: 'POST', body: fd })).json();
            if (!data.success) throw new Error(data.error || 'Upload failed');
        } else {
            fd.append('message', text);
            const data = await (await fetch('send_message.php', { method: 'POST', body: fd })).json();
            if (!data.success) throw new Error(data.error || 'Send failed');
        }

        await fetchMessages();
    } catch (err) {
        console.error('Send failed', err);
        alert('Failed to send: ' + err.message);
    } finally {
        sendBtn.disabled = false;
        msgInput.focus();
    }
}

// ── Fetch & poll ───────────────────────────────────────────────────────────
async function fetchMessages() {
    try {
        const res  = await fetch(`get_messages.php?last_id=${lastId}&viewer=${CURRENT_USER}`);
        const data = await res.json();

        if (data.messages && data.messages.length > 0) {
            const wasAtBottom =
                chatMessages.scrollHeight - chatMessages.clientHeight - chatMessages.scrollTop < 80;

            // Separate truly new vs status-only updates
            const newMsgs = data.messages.filter(m => parseInt(m.id) > lastId);
            newMsgs.forEach(m => {
                // Check for call signaling
                if (m.message && m.message.startsWith('__CALL_SIGNAL__:')) {
                    if (m.sender === OTHER_USER) {
                        const signal = m.message.split(':')[2];
                        if (signal === 'VIDEO__' || signal === 'VOICE__') {
                            isVideoCall = (signal === 'VIDEO__');
                            showCallUI('incoming');
                        } else if (signal === 'END__') {
                            endCall();
                        }
                    }
                    // Filter out signaling messages from UI
                    lastId = Math.max(lastId, parseInt(m.id));
                    return;
                }

                appendMessage(m);
                lastId = Math.max(lastId, parseInt(m.id));
            });

            // Refresh statuses on all returned messages (handles delivered→seen updates)
            refreshStatuses(data.messages);

            if (wasAtBottom && newMsgs.length) scrollToBottom();
        }
    } catch (err) {
        console.error('Fetch failed', err);
    }
}

// Initial load + poll
(async () => {
    initPeer();
    await fetchMessages();
    scrollToBottom();
    polling = setInterval(fetchMessages, 2000);
})();

window.addEventListener('beforeunload', () => clearInterval(polling));
</script>
</body>
</html>
